#include "config_xor.h"

#define USE_MD5
#define USE_SHA1

#include <stdio.h>
#include <string.h>

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
#include <sys/mman.h>
#endif

#include "sh_tiger.h"

#include "sh_unix.h"
#include "sh_error.h"
#include "sh_utils.h"

#define PRIV_MAX  32768

#if defined(TIGER_64_BIT)
#if defined(HAVE_LONG_64)
typedef unsigned long int word64;
#elif defined(HAVE_LONG_LONG_64)
typedef unsigned long long int word64;
#else
#error No 64 bit type found !
#endif
#endif

#if defined(HAVE_INT_32)
typedef unsigned int sh_word32;
#define MYFORMAT   (_("%08X%08X%08X%08X%08X%08X"))
#define GPGFORMAT (_("%08X %08X %08X  %08X %08X %08X"))
#elif defined(HAVE_LONG_32)
typedef unsigned long sh_word32;
#define MYFORMAT   (_("%08lX%08lX%08lX%08lX%08lX%08lX"))
#define GPGFORMAT (_("%08lX %08lX %08lX  %08lX %08lX %08lX"))
#elif defined(HAVE_SHORT_32)
typedef unsigned short sh_word32;
#define MYFORMAT   (_("%08X%08X%08X%08X%08X%08X"))
#define GPGFORMAT (_("%08X %08X %08X  %08X %08X %08X"))
#else
#error No 32 bit type found !
#endif

typedef unsigned char sh_byte;

#undef  FIL__
#define FIL__  _("sh_tiger0.c")

#if defined(TIGER_64_BIT)

void tiger_t(word64 *str, word64 length, word64 * res);
void tiger(word64 *str, word64 length, word64 * res);

#ifdef TIGER_DBG
static void tiger_dbg(word64 res[3], int step, 
		      unsigned long nblocks, unsigned long ncount)
{
  return;
}
#endif
#else
void tiger(sh_word32 *str, sh_word32 length, sh_word32 * res);
void tiger_t(sh_word32 *str, sh_word32 length, sh_word32 * res);

#ifdef TIGER_DBG
static 
void tiger_dbg(sh_word32 res[6], int step, 
	       unsigned long nblocks, unsigned long ncount)
{
    fprintf(stderr,                                     
            _("ST %d BLK %2ld CT %2ld %08lX %08lX %08lX %08lX %08lX %08lX\n"),
	    step,
	    nblocks,
	    ncount,
            (sh_word32)(res[1]), 
            (sh_word32)(res[0]), 
            (sh_word32)(res[3]), 
            (sh_word32)(res[2]), 
            (sh_word32)(res[5]), 
            (sh_word32)(res[4]) );
}
#endif
#endif

/* this is the wrapper function -- not part of the tiger reference
 * implementation
 */
SL_TICKET tiger_fd = (-1);

static sh_byte buffer[PRIV_MAX + 72];

#if defined(TIGER_64_BIT)
static
word64 * sh_tiger_hash_val (char * filename, TigerType what, 
			    UINT64 Length, int timeout)
#else
static
sh_word32 * sh_tiger_hash_val (char * filename, TigerType what, 
			       UINT64 Length, int timeout)
#endif
{
  SL_TICKET  fd;
  int  i, j, tt;
  int  count = 0;
  int  blk;
  char    * tmp;
  sh_byte * bptr;
  sh_byte   bbuf[64];
  UINT64    bcount = 0;

  static int lockflag = SL_FALSE;

  unsigned long pages_read;
  uid_t   euid;

  unsigned long ncount = 0, nblocks = 0;
  unsigned long  t, msb, lsb;

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE)
  /*@-nestedextern@*/
  extern long IO_Limit;
  /*@+nestedextern@*/
#endif

#if defined(TIGER_64_BIT)
#define TIGER_CAST (word64*)
  static word64 res[3];
  res[0]= (word64) 0x0123456789ABCDEFLL;
  res[1]= (word64) 0xFEDCBA9876543210LL;
  res[2]= (word64) 0xF096A5B4C3B2E187LL;
#else
#define TIGER_CAST (sh_word32*)
  static sh_word32 res[6];
  res[0]= (sh_word32) 0x89ABCDEF;
  res[1]= (sh_word32) 0x01234567;
  res[2]= (sh_word32) 0x76543210;
  res[3]= (sh_word32) 0xFEDCBA98;
  res[4]= (sh_word32) 0xC3B2E187;
  res[5]= (sh_word32) 0xF096A5B4;
#endif

  SL_ENTER(_("sh_tiger_hash_val"));

  if (what == TIGER_FD || what == TIGER_FILE) 
    {
      if (what == TIGER_FD)
	{
	  fd = tiger_fd;
	  TPT((0,FIL__, __LINE__, _("msg=<TIGER_FD>, fd=<%ld>\n"), tiger_fd));
	}
      else
	{
	  TPT((0,FIL__, __LINE__, _("msg=<TIGER_FILE>, path=<%s>\n"),
	       (filename == NULL ? _("(null)") : filename) ));
	  fd = sl_open_read (filename, SL_YESPRIV);
	}

      if (SL_ISERROR (fd)) 
	{
	  TPT((0, FIL__, __LINE__, _("msg=<SL_ISERROR (%ld)>\n"), tiger_fd));
	  tmp = sh_util_safe_name (filename);
	  (void) sl_get_euid(&euid);
	  sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, (int)fd,
			   MSG_E_ACCESS, (long) euid, tmp);
	  SH_FREE(tmp);
	  SL_RETURN( NULL, _("sh_tiger_hash_val"));
	}
      
#if defined(HAVE_MLOCK) && !defined(HAVE_BROKEN_MLOCK)
    if ((lockflag == SL_FALSE) && (skey->mlock_failed == SL_FALSE)) 
      {
	lockflag = SL_TRUE;
        if ( (-1) == sh_unix_mlock(FIL__, __LINE__, 
				   (char *)buffer, 
				   (PRIV_MAX)*sizeof(sh_byte))) 
          skey->mlock_failed = SL_TRUE;
      }
#else
    if (lockflag == SL_FALSE && skey->mlock_failed == SL_FALSE)
      {
	lockflag = SL_TRUE;
	skey->mlock_failed = SL_TRUE;
      }
#endif

#ifdef TIGER_DBG
    tiger_dbg (res, 0, nblocks, ncount);
#endif

    pages_read = 0;

    while (1) 
      {
	if (timeout > 0)
	  count = sl_read_timeout (fd, buffer, PRIV_MAX, timeout);
	else
	  count = sl_read         (fd, buffer, PRIV_MAX);

	++pages_read;

	if (SL_ISERROR (count)) 
	  {
	    if (sig_termfast == 1) {
	      SL_RETURN( NULL, _("sh_tiger_hash_val"));
	    }
	    TPT((0, FIL__ , __LINE__ , _("msg=<SL_ISERROR (%ld)>\n"), count)); 
	    tmp = sh_util_safe_name (filename);
	    if (count == SL_TIMEOUT)
	      {
		if (timeout != 7) {
		  sh_error_handle (SH_ERR_ERR, FIL__, __LINE__, count, 
				   MSG_E_TIMEOUT, timeout, tmp);
		}
	      }
	    else
	      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, 
			       count, MSG_E_READ, tmp);
	    SH_FREE(tmp);
	    memset (bbuf,   0, 64);
	    memset (buffer, 0, PRIV_MAX);

	    SL_RETURN( NULL, _("sh_tiger_hash_val"));
	  }

	if (Length > 0)
	  {
	    bcount += count;
	    if (bcount > Length) 
	      count = count - (bcount - Length);
	    count = (count < 0) ? 0 : count;
	  }

	blk      = (count / 64); /* number of 64-byte words */

	/* nblocks += blk; number of 64-byte words 
	 * count cannot be negative here, see 'if (SL_ISERROR (count))'
	 */
	tt = blk*64;

	ncount = (unsigned long) (count - tt);

	nblocks += blk;
	sh.statistics.bytes_hashed += tt;
	
	bptr = buffer; tt = 0;
	for (i = 0; i < blk; ++i)
	  {
	    bptr = &buffer[tt]; tt += 64;
	    
	    tiger_t(TIGER_CAST bptr, 64, res);
	    
#ifdef TIGER_DBG
	    tiger_dbg (res, 3, nblocks, ncount);
#endif
	  }
	
	if (blk < (PRIV_MAX / 64)) /* this must be (PRIV_MAX / 64) */
	  break;

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
	if (sig_termfast == 1) 
	  {
	    memset (bbuf,   0, 64);
	    memset (buffer, 0, PRIV_MAX);

	    SL_RETURN( NULL, _("sh_tiger_hash_val"));
	  }
	if ((IO_Limit) > 0 && (pages_read == 32)) /* check for I/O limit */
	  {
	    sh_unix_io_pause ();
	    pages_read = 0;
	  }
#endif 
      }

    TPT((0, FIL__, __LINE__ , _("msg=<Last block.>\n")));

    /* copy incomplete block
     */
    j = 0; 
    for (i = 0; i < 64; i += 4) 
      {
	bbuf[i]   = (sh_byte) '\0';
	bbuf[i+1] = (sh_byte) '\0';
	bbuf[i+2] = (sh_byte) '\0';
	bbuf[i+3] = (sh_byte) '\0';
      }
    for (i = (count/64) * 64; i < count; ++i)
      /*@-usedef@*/bbuf[j++] = buffer[i];/*@+usedef@*/

#ifdef TIGER_DBG
    tiger_dbg (res, 5, nblocks, ncount);
#endif

    msb = 0;
    t = nblocks;
    if( (lsb = t << 6) < t )    
      msb++;
    msb += t >> 26;
    t = lsb;
    if( (lsb = t + ncount) < t ) 
      msb++;
    t = lsb;
    if( (lsb = t << 3) < t )    
      msb++;
    msb += t >> 29;

    if( j < 56 ) 
      { 
        bbuf[j++] = (sh_byte) 0x01; ++ncount;
        while( j < 56 )
	  { bbuf[j++] = (sh_byte) 0; ++ncount; } 
      }
    else 
      { 
        bbuf[j++] = (sh_byte) 0x01;
        while( j < 64 )
	  bbuf[j++] = (sh_byte) 0;
	tiger_t(TIGER_CAST bbuf, 64, res);
	sh.statistics.bytes_hashed += 64;
	++nblocks; ncount = 0;
	for (i = 0; i < 56; i += 4) 
	  {
	    bbuf[i]   = (sh_byte) '\0';
	    bbuf[i+1] = (sh_byte) '\0';
	    bbuf[i+2] = (sh_byte) '\0';
	    bbuf[i+3] = (sh_byte) '\0';
	  }
        /* memset(bbuf, 0, 56 ); */ 
      }

#ifdef TIGER_DBG
    tiger_dbg (res, 6, nblocks, ncount);
#endif

    bbuf[56] = (sh_byte) (lsb      );
    bbuf[57] = (sh_byte) (lsb >>  8);
    bbuf[58] = (sh_byte) (lsb >> 16);
    bbuf[59] = (sh_byte) (lsb >> 24);
    bbuf[60] = (sh_byte) (msb      );
    bbuf[61] = (sh_byte) (msb >>  8);
    bbuf[62] = (sh_byte) (msb >> 16);
    bbuf[63] = (sh_byte) (msb >> 24);

    tiger_t(TIGER_CAST bbuf, 64, res);
    sh.statistics.bytes_hashed += 64;
    
#ifdef TIGER_DBG
    tiger_dbg (res, 7, nblocks, ncount);
#endif

    for (i = 0; i < 64; i += 4) 
      {
	bbuf[i]   = (sh_byte) '\0';
	bbuf[i+1] = (sh_byte) '\0';
	bbuf[i+2] = (sh_byte) '\0';
	bbuf[i+3] = (sh_byte) '\0';
      }

    bptr = buffer;

    memcpy(bptr, bbuf,     64); bptr +=    64;
    memcpy(bptr, bbuf,     64); bptr +=    64;
    memcpy(bptr, buffer,  128); bptr +=   128;
    memcpy(bptr, buffer,  256); bptr +=   256;
    memcpy(bptr, buffer,  512); bptr +=   512;
    memcpy(bptr, buffer, 1024); bptr +=  1024;
    memcpy(bptr, buffer, 2048); bptr +=  2048;
    memcpy(bptr, buffer, 4096); bptr +=  4096;
    memcpy(bptr, buffer, 8192); bptr +=  8192;
    memcpy(bptr, buffer,16384);

    if (what == TIGER_FILE)
      (void) sl_close (fd);
    else
      tiger_fd = (-1);


    SL_RETURN( res, _("sh_tiger_hash_val"));
  }

  if (what == TIGER_DATA && filename != NULL) 
    {
      tiger(TIGER_CAST filename, (sh_word32) Length, res); 
      SL_RETURN(res, _("sh_tiger_hash_val"));
    }
  SL_RETURN( NULL, _("sh_tiger_hash_val"));
}

/* Thu Oct 18 18:53:33 CEST 2001
 */

#ifdef USE_MD5
/*@-type@*/
/************************************************************************
 *
 *  md5.h - Declaration of functions and data types used for MD5 sum
 *  computing library functions. 
 *
 ************************************************************************/

/* Written Bob Deblier <bob@virtualunlimited.com>         */
/* Hacked to work with samhain by R. Wichmann             */
/* Need for 64bit type removed, fix for Mac OS X compiler */

typedef sh_word32     uint32;
typedef unsigned char uint8;





/* Structure to save state of computation between the single steps.  */
typedef struct
{
        uint32 h[4];
        uint32 data[16];
        uint8  offset;
        uint32  nblocks;
        int  count;
} md5Param;

static uint32 md5hinit[4] = { 0x67452301, 0xefcdab89, 0x98badcfe, 0x10325476 };

 
int md5Reset(register md5Param* p)
{
        unsigned int i;
        memcpy(p->h, md5hinit, 16);
	
        for (i = 0; i < 16; i += 8)
	  {
	    p->data[i]   = 0x00;
	    p->data[i+1] = 0x00;
	    p->data[i+2] = 0x00;
	    p->data[i+3] = 0x00;
	    p->data[i+4] = 0x00;
	    p->data[i+5] = 0x00;
	    p->data[i+6] = 0x00;
	    p->data[i+7] = 0x00;
	  }
	
        /* memset(p->data, 0x00, 64); */
        p->offset = (uint8) 0;
	p->nblocks = 0;
        return 0;
}

#if defined(__GNUC__) && defined(__i386__)
static inline UINT32
ROTL32( UINT32 x, int s)
{
        __asm__("roll %%cl,%0"
                :"=r" (x)
                :"0" (x),"c" (s));
        return x;
}
#else
#define ROTL32(x, s) (((x) << (s)) | ((x) >> (32 - (s))))
#endif


#define FF(a, b, c, d, w, s, t) \
        a += ((b&(c^d))^d) + w + t;     \
        a = ROTL32(a, s);       \
        a += b;

#define GG(a, b, c, d, w, s, t) \
        a += ((d&(b^c))^c) + w + t;     \
        a = ROTL32(a, s);       \
        a += b;

#define HH(a, b, c, d, w, s, t) \
        a += (b^c^d) + w + t;   \
        a = ROTL32(a, s);       \
        a += b;

#define II(a, b, c, d, w, s, t) \
        a += (c^(b|~d)) + w + t;        \
        a = ROTL32(a, s);       \
        a += b;

#if WORDS_BIGENDIAN
uint32 swapu32(uint32 n)
{
        return (    ((n & 0xffU) << 24) |
                                ((n & 0xff00U) << 8) |
                                ((n & 0xff0000U) >> 8) |
                                ((n & 0xff000000U) >> 24) );
}
#endif

static
void md5Process(md5Param* p)
{
        register uint32 a,b,c,d;
        register uint32* w;
        #if WORDS_BIGENDIAN
        register sh_byte t;
        #endif

        w = p->data;
        #if WORDS_BIGENDIAN
        t = 16;
        while (t--)
        {
                register uint32 temp = swapu32(*w);
                *(w++) = temp;
        }
        w = p->data;
        #endif

        a = p->h[0]; b = p->h[1]; c = p->h[2]; d = p->h[3];

        FF(a, b, c, d, (*w++),  7, 0xd76aa478);
        FF(d, a, b, c, (*w++), 12, 0xe8c7b756);
        FF(c, d, a, b, (*w++), 17, 0x242070db);
        FF(b, c, d, a, (*w++), 22, 0xc1bdceee);
        FF(a, b, c, d, (*w++),  7, 0xf57c0faf);
        FF(d, a, b, c, (*w++), 12, 0x4787c62a);
        FF(c, d, a, b, (*w++), 17, 0xa8304613);
        FF(b, c, d, a, (*w++), 22, 0xfd469501);
        FF(a, b, c, d, (*w++),  7, 0x698098d8);
        FF(d, a, b, c, (*w++), 12, 0x8b44f7af);
        FF(c, d, a, b, (*w++), 17, 0xffff5bb1);
        FF(b, c, d, a, (*w++), 22, 0x895cd7be);
        FF(a, b, c, d, (*w++),  7, 0x6b901122);
        FF(d, a, b, c, (*w++), 12, 0xfd987193);
        FF(c, d, a, b, (*w++), 17, 0xa679438e);
        FF(b, c, d, a, (*w++), 22, 0x49b40821);

	w = p->data;

        GG(a, b, c, d, w[ 1],  5, 0xf61e2562);
        GG(d, a, b, c, w[ 6],  9, 0xc040b340);
        GG(c, d, a, b, w[11], 14, 0x265e5a51);
        GG(b, c, d, a, w[ 0], 20, 0xe9b6c7aa);
        GG(a, b, c, d, w[ 5],  5, 0xd62f105d);
        GG(d, a, b, c, w[10],  9, 0x02441453);
        GG(c, d, a, b, w[15], 14, 0xd8a1e681);
        GG(b, c, d, a, w[ 4], 20, 0xe7d3fbc8);
        GG(a, b, c, d, w[ 9],  5, 0x21e1cde6);
        GG(d, a, b, c, w[14],  9, 0xc33707d6);
        GG(c, d, a, b, w[ 3], 14, 0xf4d50d87);
        GG(b, c, d, a, w[ 8], 20, 0x455a14ed);
        GG(a, b, c, d, w[13],  5, 0xa9e3e905);
        GG(d, a, b, c, w[ 2],  9, 0xfcefa3f8);
        GG(c, d, a, b, w[ 7], 14, 0x676f02d9);
        GG(b, c, d, a, w[12], 20, 0x8d2a4c8a);

        HH(a, b, c, d, w[ 5],  4, 0xfffa3942);
        HH(d, a, b, c, w[ 8], 11, 0x8771f681);
        HH(c, d, a, b, w[11], 16, 0x6d9d6122);
        HH(b, c, d, a, w[14], 23, 0xfde5380c);
        HH(a, b, c, d, w[ 1],  4, 0xa4beea44);
        HH(d, a, b, c, w[ 4], 11, 0x4bdecfa9);
        HH(c, d, a, b, w[ 7], 16, 0xf6bb4b60);
        HH(b, c, d, a, w[10], 23, 0xbebfbc70);
        HH(a, b, c, d, w[13],  4, 0x289b7ec6);
        HH(d, a, b, c, w[ 0], 11, 0xeaa127fa);
        HH(c, d, a, b, w[ 3], 16, 0xd4ef3085);
        HH(b, c, d, a, w[ 6], 23, 0x04881d05);
        HH(a, b, c, d, w[ 9],  4, 0xd9d4d039);
        HH(d, a, b, c, w[12], 11, 0xe6db99e5);
        HH(c, d, a, b, w[15], 16, 0x1fa27cf8);
        HH(b, c, d, a, w[ 2], 23, 0xc4ac5665);

        II(a, b, c, d, w[ 0],  6, 0xf4292244);
        II(d, a, b, c, w[ 7], 10, 0x432aff97);
        II(c, d, a, b, w[14], 15, 0xab9423a7);
        II(b, c, d, a, w[ 5], 21, 0xfc93a039);
        II(a, b, c, d, w[12],  6, 0x655b59c3);
        II(d, a, b, c, w[ 3], 10, 0x8f0ccc92);
        II(c, d, a, b, w[10], 15, 0xffeff47d);
        II(b, c, d, a, w[ 1], 21, 0x85845dd1);
        II(a, b, c, d, w[ 8],  6, 0x6fa87e4f);
        II(d, a, b, c, w[15], 10, 0xfe2ce6e0);
        II(c, d, a, b, w[ 6], 15, 0xa3014314);
        II(b, c, d, a, w[13], 21, 0x4e0811a1);
        II(a, b, c, d, w[ 4],  6, 0xf7537e82);
        II(d, a, b, c, w[11], 10, 0xbd3af235);
        II(c, d, a, b, w[ 2], 15, 0x2ad7d2bb);
        II(b, c, d, a, w[ 9], 21, 0xeb86d391);

        p->h[0] += a;
        p->h[1] += b;
        p->h[2] += c;
        p->h[3] += d;
}

int md5Update(md5Param* p, const sh_byte* data, int size)
{
        register int proclength;

        while (size > 0)
        {
	  proclength = (((int)p->offset + size) > 64) ? 
	    (64 - (int)p->offset) : size;
	  memcpy(((sh_byte *) p->data) + p->offset, data, (size_t) proclength);
	  size -= proclength;
	  data += proclength;
	  p->offset += proclength;
	  
	  if (p->offset == (uint8) 64)
	    {
	      md5Process(p);
	      p->offset = (uint8) 0;
	      p->nblocks++;
	    }
        }
        return 0;
}

static void md5Finish(md5Param* p)
{
        uint32 t, msb, lsb;
	uint8 * pp;
        register uint8 *ptr;

        msb = 0;
	t = p->nblocks;
	if( (lsb = t << 6) < t ) /* multiply by 64 to make a byte count */
	  msb++;
	msb += t >> 26;
	t = lsb;
	if( (lsb = t + (uint32)p->offset) < t ) /* add the count */
	  msb++;
	t = lsb;
	if( (lsb = t << 3) < t ) /* multiply by 8 to make a bit count */
	  msb++;
	msb += t >> 29;

	ptr = ((uint8 *) p->data) + p->offset++;

 
        *(ptr++) = (uint8) 0x80;

        if (p->offset > (uint8)56)
        {
                while (p->offset++ < 64)
                        *(ptr++) = 0;

                md5Process(p);
                p->offset = 0;
        }

        ptr = ((uint8 *) p->data) + p->offset;
        while (p->offset++ < 56)
                *(ptr++) = 0;

	/* append the 64 bit count */
	*(ptr++) = lsb	   ;
	*(ptr++) = lsb >>  8;
	*(ptr++) = lsb >> 16;
	*(ptr++) = lsb >> 24;
	*(ptr++) = msb	   ;
	*(ptr++) = msb >>  8;
	*(ptr++) = msb >> 16;
	*(ptr++) = msb >> 24;

        md5Process(p);

	pp = (uint8 *) p->data;
#ifdef WORDS_BIGENDIAN
#define X(a) do { *pp++ = (*p).a; *pp++ = (*p).a >> 8;      \
		  *pp++ = (*p).a >> 16; *pp++ = (*p).a >> 24; } while(0)
#else /* little endian */
    /*#define X(a) do { *(uint32*)p = p->##a ; p += 4; } while(0)*/
    /* Unixware's cpp doesn't like the above construct so we do it his way:
     * (reported by Allan Clark) */
#define X(a) do { *(uint32*)pp = (*p).a ; pp += 4; } while(0)
#endif
	X(h[0]);
	X(h[1]);
	X(h[2]);
	X(h[3]);
#undef X

        p->offset = 0;
}

int md5Digest(md5Param* p, uint32* data)
{
        md5Finish(p);
        memcpy(data, p->h, 16);
        (void) md5Reset(p);
        return 0;
}
/*@+type@*/

/* Compute MD5 message digest for bytes read from STREAM.  The
   resulting message digest number will be written into the 16 bytes
   beginning at RESBLOCK.  */
static int md5_stream(char * filename, void *resblock, 
		      UINT64 Length, int timeout)
{
  /* Important: BLOCKSIZE must be a multiple of 64.  */
  static const int BLOCKSIZE = 8192;
  md5Param ctx;
  char buffer[8264]; /* BLOCKSIZE + 72  AIX compiler chokes */
  off_t sum = 0;
  SL_TICKET  fd;
  char * tmp;
  uid_t   euid;
  UINT64  bcount = 0;

  unsigned long pages_read;
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  /*@-nestedextern@*/
  extern long IO_Limit;
  /*@+nestedextern@*/
#endif

  /* Initialize the computation context.  */
  (void) md5Reset (&ctx);

  fd = tiger_fd;

  if (SL_ISERROR (fd))
    {
      TPT((0, FIL__, __LINE__, _("msg=<SL_ISERROR (%ld)>\n"), tiger_fd));
      tmp = sh_util_safe_name (filename);
      (void) sl_get_euid(&euid);
      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, fd,
		       MSG_E_ACCESS, (long) euid, tmp);
      SH_FREE(tmp);
      return -1;
    }

  pages_read = 0;

  /* Iterate over full file contents.  */
  while (1 == 1) {
    /* We read the file in blocks of BLOCKSIZE bytes.  One call of the
       computation function processes the whole buffer so that with the
       next round of the loop another block can be read.  */
    off_t  n;
    sum = 0;

    /* Read block.  Take care for partial reads.  */
    do {

      n = (off_t) sl_read_timeout (fd, buffer + sum, 
				   (size_t) BLOCKSIZE - sum, timeout);

      if (SL_ISERROR (n))
	{
	  if (sig_termfast == 1)
	    return -1;
	  TPT((0, FIL__ , __LINE__ , _("msg=<SL_ISERROR (%ld)>\n"), n));
	  tmp = sh_util_safe_name (filename);
	  if (n == SL_TIMEOUT) 
	    {
	      if (timeout != 7) {
		sh_error_handle (SH_ERR_ERR, FIL__, __LINE__, n, MSG_E_TIMEOUT,
				 timeout, tmp);
	      }
	    }
	  else
	    sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, n,
			     MSG_E_READ, tmp);
	  SH_FREE(tmp);
	  return -1;
	}

      if (Length > 0)
	{
	  bcount += n;
	  if (bcount > Length) 
	    n = n - (bcount - Length);
	  n = (n < 0) ? 0 : n;
	}

      sum += n;
    }
    while (sum < (off_t) BLOCKSIZE 
	   && n != 0);

    ++pages_read;

    /* If end of file is reached, end the loop.  */
    if (n == 0)
      break;

    /* Process buffer with BLOCKSIZE bytes.  Note that
       BLOCKSIZE % 64 == 0
    */
    (void) md5Update(&ctx, (sh_byte*) buffer, BLOCKSIZE);
    sh.statistics.bytes_hashed += BLOCKSIZE;

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
    if ((IO_Limit) > 0 && (pages_read == 32)) /* check for I/O limit */
      {
	sh_unix_io_pause ();
	pages_read = 0;
      }
    if (sig_termfast == 1) 
      {
	return -1;
      }
#endif 
  }

  /* Add the last bytes if necessary.  */
  if (sum > 0)
    {
      (void) md5Update(&ctx, (sh_byte*) buffer, (int) sum);
      sh.statistics.bytes_hashed += BLOCKSIZE;
    }

  /* Construct result in desired memory.  */
  (void) md5Digest(&ctx, resblock);

  return 0;
}

static
char * sh_tiger_md5_hash  (char * filename, TigerType what, 
			   UINT64 Length, int timeout)
{
  int cnt;
  static char out[KEY_LEN+1];
  unsigned char md5buffer[16];

  if (what != TIGER_FD)
    {
      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, 0,
		       MSG_E_SUBGEN, _("Not TIGER_FD"), 
		       _("sh_tiger_md5_hash"));
      out[0] = '\0';
      return out;
    }

  (void) md5_stream (filename, md5buffer, Length, timeout);

  /*@-bufferoverflowhigh -usedef@*/
  for (cnt = 0; cnt < 16; ++cnt)
    sprintf (&out[cnt*2], _("%02X"),                 /* known to fit  */
	     (unsigned int) md5buffer[cnt]);
  /*@+bufferoverflowhigh +usedef@*/
  for (cnt = 32; cnt < KEY_LEN; ++cnt)
    out[cnt] = '0';
  out[KEY_LEN] = '\0';

  return out;
}

/* USE_MD5 */
#endif

/***************************************************************
 *
 * SHA1
 *
 ***************************************************************/

#ifdef USE_SHA1
/*@-type@*/

typedef unsigned char sha_word8;
typedef sh_word32     sha_word32;

/* The SHA block size and message digest sizes, in bytes */

#define SHA_DATASIZE    64
#define SHA_DATALEN     16
#define SHA_DIGESTSIZE  20
#define SHA_DIGESTLEN    5
/* The structure for storing SHA info */

typedef struct sha_ctx {
  sha_word32 digest[SHA_DIGESTLEN];  /* Message digest */
  sha_word32 count_l, count_h;       /* 64-bit block count */
  sha_word8 block[SHA_DATASIZE];     /* SHA data buffer */
  int index;                             /* index into buffer */
} SHA_CTX;

static void sha_init(struct sha_ctx *ctx);
static void sha_update(struct sha_ctx *ctx, sha_word8 *buffer,sha_word32 len);
static void sha_final(struct sha_ctx *ctx);
static void sha_digest(struct sha_ctx *ctx, sha_word8 *s);


/* The SHA f()-functions.  The f1 and f3 functions can be optimized to
   save one boolean operation each - thanks to Rich Schroeppel,
   rcs@cs.arizona.edu for discovering this */

/*#define f1(x,y,z) ( ( x & y ) | ( ~x & z ) )          // Rounds  0-19 */
#define f1(x,y,z)   ( z ^ ( x & ( y ^ z ) ) )           /* Rounds  0-19 */
#define f2(x,y,z)   ( x ^ y ^ z )                       /* Rounds 20-39 */
/*#define f3(x,y,z) ( ( x & y ) | ( x & z ) | ( y & z ) )   // Rounds 40-59 */
#define f3(x,y,z)   ( ( x & y ) | ( z & ( x | y ) ) )   /* Rounds 40-59 */
#define f4(x,y,z)   ( x ^ y ^ z )                       /* Rounds 60-79 */

/* The SHA Mysterious Constants */

#define K1  0x5A827999L                                 /* Rounds  0-19 */
#define K2  0x6ED9EBA1L                                 /* Rounds 20-39 */
#define K3  0x8F1BBCDCL                                 /* Rounds 40-59 */
#define K4  0xCA62C1D6L                                 /* Rounds 60-79 */

/* SHA initial values */

#define h0init  0x67452301L
#define h1init  0xEFCDAB89L
#define h2init  0x98BADCFEL
#define h3init  0x10325476L
#define h4init  0xC3D2E1F0L

/* 32-bit rotate left - kludged with shifts */

#define ROTL(n,X)  ( ( (X) << (n) ) | ( (X) >> ( 32 - (n) ) ) )

/* The initial expanding function.  The hash function is defined over an
   80-word expanded input array W, where the first 16 are copies of the input
   data, and the remaining 64 are defined by

        W[ i ] = W[ i - 16 ] ^ W[ i - 14 ] ^ W[ i - 8 ] ^ W[ i - 3 ]

   This implementation generates these values on the fly in a circular
   buffer - thanks to Colin Plumb, colin@nyx10.cs.du.edu for this
   optimization.

   The updated SHA changes the expanding function by adding a rotate of 1
   bit.  Thanks to Jim Gillogly, jim@rand.org, and an anonymous contributor
   for this information */

#define expand(W,i) ( W[ i & 15 ] = \
                      ROTL( 1, ( W[ i & 15 ] ^ W[ (i - 14) & 15 ] ^ \
                                 W[ (i - 8) & 15 ] ^ W[ (i - 3) & 15 ] ) ) )


/* The prototype SHA sub-round.  The fundamental sub-round is:

        a' = e + ROTL( 5, a ) + f( b, c, d ) + k + data;
        b' = a;
        c' = ROTL( 30, b );
        d' = c;
        e' = d;

   but this is implemented by unrolling the loop 5 times and renaming the
   variables ( e, a, b, c, d ) = ( a', b', c', d', e' ) each iteration.
   This code is then replicated 20 times for each of the 4 functions, using
   the next 20 values from the W[] array each time */

#define subRound(a, b, c, d, e, f, k, data) \
    ( e += ROTL( 5, a ) + f( b, c, d ) + k + data, b = ROTL( 30, b ) )

/* Initialize the SHA values */

static void sha_init(struct sha_ctx *ctx)
{
  /* Set the h-vars to their initial values */
  ctx->digest[ 0 ] = h0init;
  ctx->digest[ 1 ] = h1init;
  ctx->digest[ 2 ] = h2init;
  ctx->digest[ 3 ] = h3init;
  ctx->digest[ 4 ] = h4init;

  /* Initialize bit count */
  ctx->count_l = ctx->count_h = 0;

  /* Initialize buffer */
  ctx->index = 0;
}

/* Perform the SHA transformation.  Note that this code, like MD5, seems to
   break some optimizing compilers due to the complexity of the expressions
   and the size of the basic block.  It may be necessary to split it into
   sections, e.g. based on the four subrounds

   Note that this function destroys the data area */

static void sha_transform(struct sha_ctx *ctx, sha_word32 *data )
{
  register sha_word32 A, B, C, D, E;     /* Local vars */

  /* Set up first buffer and local data buffer */
  A = ctx->digest[0];
  B = ctx->digest[1];
  C = ctx->digest[2];
  D = ctx->digest[3];
  E = ctx->digest[4];

  /* Heavy mangling, in 4 sub-rounds of 20 interations each. */
  subRound( A, B, C, D, E, f1, K1, data[ 0] );
  subRound( E, A, B, C, D, f1, K1, data[ 1] );
  subRound( D, E, A, B, C, f1, K1, data[ 2] );
  subRound( C, D, E, A, B, f1, K1, data[ 3] );
  subRound( B, C, D, E, A, f1, K1, data[ 4] );
  subRound( A, B, C, D, E, f1, K1, data[ 5] );
  subRound( E, A, B, C, D, f1, K1, data[ 6] );
  subRound( D, E, A, B, C, f1, K1, data[ 7] );
  subRound( C, D, E, A, B, f1, K1, data[ 8] );
  subRound( B, C, D, E, A, f1, K1, data[ 9] );
  subRound( A, B, C, D, E, f1, K1, data[10] );
  subRound( E, A, B, C, D, f1, K1, data[11] );
  subRound( D, E, A, B, C, f1, K1, data[12] );
  subRound( C, D, E, A, B, f1, K1, data[13] );
  subRound( B, C, D, E, A, f1, K1, data[14] );
  subRound( A, B, C, D, E, f1, K1, data[15] );
  subRound( E, A, B, C, D, f1, K1, expand( data, 16 ) );
  subRound( D, E, A, B, C, f1, K1, expand( data, 17 ) );
  subRound( C, D, E, A, B, f1, K1, expand( data, 18 ) );
  subRound( B, C, D, E, A, f1, K1, expand( data, 19 ) );

  subRound( A, B, C, D, E, f2, K2, expand( data, 20 ) );
  subRound( E, A, B, C, D, f2, K2, expand( data, 21 ) );
  subRound( D, E, A, B, C, f2, K2, expand( data, 22 ) );
  subRound( C, D, E, A, B, f2, K2, expand( data, 23 ) );
  subRound( B, C, D, E, A, f2, K2, expand( data, 24 ) );
  subRound( A, B, C, D, E, f2, K2, expand( data, 25 ) );
  subRound( E, A, B, C, D, f2, K2, expand( data, 26 ) );
  subRound( D, E, A, B, C, f2, K2, expand( data, 27 ) );
  subRound( C, D, E, A, B, f2, K2, expand( data, 28 ) );
  subRound( B, C, D, E, A, f2, K2, expand( data, 29 ) );
  subRound( A, B, C, D, E, f2, K2, expand( data, 30 ) );
  subRound( E, A, B, C, D, f2, K2, expand( data, 31 ) );
  subRound( D, E, A, B, C, f2, K2, expand( data, 32 ) );
  subRound( C, D, E, A, B, f2, K2, expand( data, 33 ) );
  subRound( B, C, D, E, A, f2, K2, expand( data, 34 ) );
  subRound( A, B, C, D, E, f2, K2, expand( data, 35 ) );
  subRound( E, A, B, C, D, f2, K2, expand( data, 36 ) );
  subRound( D, E, A, B, C, f2, K2, expand( data, 37 ) );
  subRound( C, D, E, A, B, f2, K2, expand( data, 38 ) );
  subRound( B, C, D, E, A, f2, K2, expand( data, 39 ) );

  subRound( A, B, C, D, E, f3, K3, expand( data, 40 ) );
  subRound( E, A, B, C, D, f3, K3, expand( data, 41 ) );
  subRound( D, E, A, B, C, f3, K3, expand( data, 42 ) );
  subRound( C, D, E, A, B, f3, K3, expand( data, 43 ) );
  subRound( B, C, D, E, A, f3, K3, expand( data, 44 ) );
  subRound( A, B, C, D, E, f3, K3, expand( data, 45 ) );
  subRound( E, A, B, C, D, f3, K3, expand( data, 46 ) );
  subRound( D, E, A, B, C, f3, K3, expand( data, 47 ) );
  subRound( C, D, E, A, B, f3, K3, expand( data, 48 ) );
  subRound( B, C, D, E, A, f3, K3, expand( data, 49 ) );
  subRound( A, B, C, D, E, f3, K3, expand( data, 50 ) );
  subRound( E, A, B, C, D, f3, K3, expand( data, 51 ) );
  subRound( D, E, A, B, C, f3, K3, expand( data, 52 ) );
  subRound( C, D, E, A, B, f3, K3, expand( data, 53 ) );
  subRound( B, C, D, E, A, f3, K3, expand( data, 54 ) );
  subRound( A, B, C, D, E, f3, K3, expand( data, 55 ) );
  subRound( E, A, B, C, D, f3, K3, expand( data, 56 ) );
  subRound( D, E, A, B, C, f3, K3, expand( data, 57 ) );
  subRound( C, D, E, A, B, f3, K3, expand( data, 58 ) );
  subRound( B, C, D, E, A, f3, K3, expand( data, 59 ) );

  subRound( A, B, C, D, E, f4, K4, expand( data, 60 ) );
  subRound( E, A, B, C, D, f4, K4, expand( data, 61 ) );
  subRound( D, E, A, B, C, f4, K4, expand( data, 62 ) );
  subRound( C, D, E, A, B, f4, K4, expand( data, 63 ) );
  subRound( B, C, D, E, A, f4, K4, expand( data, 64 ) );
  subRound( A, B, C, D, E, f4, K4, expand( data, 65 ) );
  subRound( E, A, B, C, D, f4, K4, expand( data, 66 ) );
  subRound( D, E, A, B, C, f4, K4, expand( data, 67 ) );
  subRound( C, D, E, A, B, f4, K4, expand( data, 68 ) );
  subRound( B, C, D, E, A, f4, K4, expand( data, 69 ) );
  subRound( A, B, C, D, E, f4, K4, expand( data, 70 ) );
  subRound( E, A, B, C, D, f4, K4, expand( data, 71 ) );
  subRound( D, E, A, B, C, f4, K4, expand( data, 72 ) );
  subRound( C, D, E, A, B, f4, K4, expand( data, 73 ) );
  subRound( B, C, D, E, A, f4, K4, expand( data, 74 ) );
  subRound( A, B, C, D, E, f4, K4, expand( data, 75 ) );
  subRound( E, A, B, C, D, f4, K4, expand( data, 76 ) );
  subRound( D, E, A, B, C, f4, K4, expand( data, 77 ) );
  subRound( C, D, E, A, B, f4, K4, expand( data, 78 ) );
  subRound( B, C, D, E, A, f4, K4, expand( data, 79 ) );

  /* Build message digest */
  ctx->digest[0] += A;
  ctx->digest[1] += B;
  ctx->digest[2] += C;
  ctx->digest[3] += D;
  ctx->digest[4] += E;
}

#if 1

#ifndef EXTRACT_UCHAR
#define EXTRACT_UCHAR(p)  (*(unsigned char *)(p))
#endif

#define STRING2INT(s) ((((((EXTRACT_UCHAR(s) << 8)    \
                         | EXTRACT_UCHAR(s+1)) << 8)  \
                         | EXTRACT_UCHAR(s+2)) << 8)  \
                         | EXTRACT_UCHAR(s+3))
#else
sha_word32 STRING2INT(word8 *s)
{
  sha_word32 r;
  int i;

  for (i = 0, r = 0; i < 4; i++, s++)
    r = (r << 8) | *s;
  return r;
}
#endif

static void sha_block(struct sha_ctx *ctx, sha_word8 *block)
{
  sha_word32 data[SHA_DATALEN];
  int i;

  /* Update block count */
  /*@-boolops@*/
  if (!++ctx->count_l)
    ++ctx->count_h;
  /*@+boolops@*/

  /* Endian independent conversion */
  for (i = 0; i<SHA_DATALEN; i++, block += 4)
    data[i] = STRING2INT(block);

  sha_transform(ctx, data);
}

static void sha_update(struct sha_ctx *ctx, sha_word8 *buffer, sha_word32 len)
{
  if (ctx->index != 0)
    { /* Try to fill partial block */
      unsigned left = SHA_DATASIZE - ctx->index;
      if (len < left)
        {
          memmove(ctx->block + ctx->index, buffer, len);
          ctx->index += len;
          return; /* Finished */
        }
      else
        {
          memmove(ctx->block + ctx->index, buffer, left);
          sha_block(ctx, ctx->block);
          buffer += left;
          len -= left;
        }
    }
  while (len >= SHA_DATASIZE)
    {
      sha_block(ctx, buffer);
      buffer += SHA_DATASIZE;
      len -= SHA_DATASIZE;
    }
  /*@-predboolint@*/
  if ((ctx->index = len))     /* This assignment is intended */
  /*@+predboolint@*/
    /* Buffer leftovers */
    memmove(ctx->block, buffer, len);
}

/* Final wrapup - pad to SHA_DATASIZE-byte boundary with the bit pattern
   1 0* (64-bit count of bits processed, MSB-first) */

static void sha_final(struct sha_ctx *ctx)
{
  sha_word32 data[SHA_DATALEN];
  int i;
  int words;

  i = ctx->index;
  /* Set the first char of padding to 0x80.  This is safe since there is
     always at least one byte free */
  ctx->block[i++] = 0x80;

  /* Fill rest of word */
  /*@-predboolint@*/
  for( ; i & 3; i++)
    ctx->block[i] = 0;
  /*@+predboolint@*/

  /* i is now a multiple of the word size 4 */
  /*@-shiftimplementation@*/
  words = i >> 2;
  /*@+shiftimplementation@*/
  for (i = 0; i < words; i++)
    data[i] = STRING2INT(ctx->block + 4*i);

  if (words > (SHA_DATALEN-2))
    { /* No room for length in this block. Process it and
       * pad with another one */
      for (i = words ; i < SHA_DATALEN; i++)
        data[i] = 0;
      sha_transform(ctx, data);
      for (i = 0; i < (SHA_DATALEN-2); i++)
        data[i] = 0;
    }
  else
    for (i = words ; i < SHA_DATALEN - 2; i++)
      data[i] = 0;
  /* Theres 512 = 2^9 bits in one block */
  /*@-shiftimplementation@*/
  data[SHA_DATALEN-2] = (ctx->count_h << 9) | (ctx->count_l >> 23);
  data[SHA_DATALEN-1] = (ctx->count_l << 9) | (ctx->index << 3);
  /*@+shiftimplementation@*/
  sha_transform(ctx, data);
}

static void sha_digest(struct sha_ctx *ctx, sha_word8 *s)
{
  int i;

  for (i = 0; i < SHA_DIGESTLEN; i++)
    {
      *s++ =         ctx->digest[i] >> 24;
      *s++ = 0xff & (ctx->digest[i] >> 16);
      *s++ = 0xff & (ctx->digest[i] >> 8);
      *s++ = 0xff &  ctx->digest[i];
    }
}
/*@+type@*/

/* Compute SHA1 message digest for bytes read from STREAM.  The
   resulting message digest number will be written into the 16 bytes
   beginning at RESBLOCK.  */
static int sha1_stream(char * filename, void *resblock, 
		       UINT64 Length, int timeout)
{
  /* Important: BLOCKSIZE must be a multiple of 64.  */
  static const int BLOCKSIZE = 4096;
  struct sha_ctx ctx;
  char buffer[4168]; /* BLOCKSIZE + 72 AIX compiler chokes */
  off_t sum = 0;
  SL_TICKET  fd;
  char * tmp;
  uid_t  euid;
  UINT64 bcount = 0;

  unsigned long pages_read;
#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
  /*@-nestedextern@*/
  extern long IO_Limit;
  /*@+nestedextern@*/
#endif

  /* Initialize the computation context.  */
  (void) sha_init(&ctx);

  fd = tiger_fd;

  if (SL_ISERROR (fd))
    {
      TPT((0, FIL__, __LINE__, _("msg=<SL_ISERROR (%ld)>\n"), tiger_fd));
      tmp = sh_util_safe_name (filename);
      (void) sl_get_euid(&euid);
      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, fd,
		       MSG_E_ACCESS, (long) euid, tmp);
      SH_FREE(tmp);
      return -1;
    }

  /* Iterate over full file contents.  */

  pages_read = 0;

  while (1 == 1) {
    /* We read the file in blocks of BLOCKSIZE bytes.  One call of the
       computation function processes the whole buffer so that with the
       next round of the loop another block can be read.  */
    off_t  n;
    sum = 0;

    /* Read block.  Take care for partial reads.  */
    do {
      n = (off_t) sl_read_timeout(fd, buffer + sum, 
				  (size_t) BLOCKSIZE - sum, timeout);

      if (SL_ISERROR (n))
	{
	  if (sig_termfast == 1)
	    return -1;

	  TPT((0, FIL__ , __LINE__ , _("msg=<SL_ISERROR (%ld)>\n"), n));
	  tmp = sh_util_safe_name (filename);
	  if (n == SL_TIMEOUT)
	    {
	      if (timeout != 7) {
		sh_error_handle (SH_ERR_ERR, FIL__, __LINE__, n, MSG_E_TIMEOUT,
				 timeout, tmp);
	      }
	    }
	  else 
	    {
	      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, n,
			       MSG_E_READ, tmp);
	    }
	  SH_FREE(tmp);
	  return -1;
	}

      if (Length > 0)
	{
	  bcount += n;
	  if (bcount > Length) 
	    n = n - (bcount - Length);
	  n = (n < 0) ? 0 : n;
	}

      sum += n;
    }
    while (sum < (off_t)BLOCKSIZE 
	   && n != 0);

    ++pages_read;

    /* If end of file is reached, end the loop.  */
    if (n == 0)
      break;

    /* Process buffer with BLOCKSIZE bytes.  Note that
       BLOCKSIZE % 64 == 0
    */
    sha_update(&ctx, (sha_word8*) buffer, (sha_word32) BLOCKSIZE);
    sh.statistics.bytes_hashed += BLOCKSIZE;

#if defined (SH_WITH_CLIENT) || defined (SH_STANDALONE) 
    if ((IO_Limit) > 0 && (pages_read == 32)) /* check for I/O limit */
      {
	sh_unix_io_pause ();
	pages_read = 0;
      }
    if (sig_termfast == 1) 
      {
	return -1;
      }
#endif 

  }

  /* Add the last bytes if necessary.  */
  if (sum > 0)
    {
      sha_update(&ctx, (sha_word8*) buffer, (sha_word32) sum);
      sh.statistics.bytes_hashed += sum;
    }

  sha_final (&ctx);

  /* Construct result in desired memory.  */
  sha_digest (&ctx, resblock);
  return 0;
}


static char * sh_tiger_sha1_hash  (char * filename, TigerType what, 
				   UINT64 Length, int timeout)
{
  int cnt = (int) Length;  /* fix compiler warning */
  static char out[KEY_LEN+1];
  unsigned char sha1buffer[20];

  if (what != TIGER_FD)
    {
      sh_error_handle (ShDFLevel[SH_ERR_T_FILE], FIL__, __LINE__, 0,
		       MSG_E_SUBGEN, _("Not TIGER_FD"), 
		       _("sh_tiger_sha1_hash"));
      out[0] = '\0';
      return out;
    }

  (void) sha1_stream (filename, sha1buffer, Length, timeout);

  /*@-bufferoverflowhigh -usedef@*/
  for (cnt = 0; cnt < 20; ++cnt)
    sprintf (&out[cnt*2], _("%02X"),                 /* known to fit  */
	     (unsigned int) sha1buffer[cnt]);
  /*@+bufferoverflowhigh +usedef@*/
  for (cnt = 40; cnt < KEY_LEN; ++cnt)
    out[cnt] = '0';
  out[KEY_LEN] = '\0';

  return out;
}

/* ifdef USE_SHA1 */
#endif

static int hash_type = 0;

int sh_tiger_get_hashtype ()
{
  return hash_type;
}

int sh_tiger_hashtype (char * c)
{
  SL_ENTER( _("sh_tiger_hashtype"));

  if (!c)
    {
      SL_RETURN( -1, _("sh_tiger_hashtype"));
    }

  if (0 == strcmp(c, _("TIGER192")))
    hash_type = 0;
#ifdef USE_MD5
  else if (0 == strcmp(c, _("SHA1")))    
    hash_type = 1;
#endif
#ifdef USE_SHA1
  else if (0 == strcmp(c, _("MD5")))    
    hash_type = 2;
#endif
  else
    {
      SL_RETURN( -1, _("sh_tiger_hashtype"));
    }
  SL_RETURN( 0, _("sh_tiger_hashtype"));
}

static char * sh_tiger_hash_internal (char * filename, TigerType what, 
				      UINT64 Length, int timeout);

char * sh_tiger_hash (char * filename, TigerType what, 
		      UINT64 Length)
{
  return sh_tiger_hash_internal (filename, what, Length, 0);
}

char * sh_tiger_generic_hash (char * filename, TigerType what, 
			      UINT64 Length, int timeout)
{
#ifdef USE_SHA1
  if (hash_type == 1)
    return sh_tiger_sha1_hash    (filename, what, Length, timeout);
#endif
#ifdef USE_MD5
  if (hash_type == 2)
    return sh_tiger_md5_hash     (filename, what, Length, timeout);
#endif
  return sh_tiger_hash_internal  (filename, what, Length, timeout);
}

/*
 * -------   end new ---------  */
  
static char * sh_tiger_hash_internal (char * filename, TigerType what, 
				      UINT64 Length, int timeout)
{
#if defined(TIGER_64_BIT)
  word64 * res;
#else
  sh_word32 * res;
#endif
  static char out[KEY_LEN+1];

  SL_ENTER( _("sh_tiger_hash_internal"));

  res = sh_tiger_hash_val (filename, what, Length, timeout);

  if (res != NULL)
    {
      /*@-bufferoverflowhigh -formatconst@*/
#if defined(TIGER_64_BIT)
      sprintf(out,                                   /* known to fit  */
	      MYFORMAT,
	      (sh_word32)(res[0]>>32), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[1]>>32), 
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[2]>>32), 
	      (sh_word32)(res[2]) );
#else
      sprintf(out,                                   /* known to fit  */
	      MYFORMAT,
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[3]), 
	      (sh_word32)(res[2]), 
	      (sh_word32)(res[5]), 
	      (sh_word32)(res[4]) );
#endif
      /*@+bufferoverflowhigh@*/
      out[KEY_LEN] = '\0';
      SL_RETURN( out, _("sh_tiger_hash_internal"));

    }

   SL_RETURN( _("000000000000000000000000000000000000000000000000"), 
	      _("sh_tiger_hash_internal"));
}

char * sh_tiger_hash_gpg (char * filename, TigerType what, 
			  UINT64 Length)
{
  size_t  len;
  char * out;
  char   outhash[48+6+1];
#if defined(TIGER_64_BIT)
  word64 * res;
#else
  sh_word32 * res;
#endif

  SL_ENTER(_("sh_tiger_hash_gpg"));

  res = sh_tiger_hash_val (filename, what, Length, 0);
  if (res != NULL)
    {
      /*@-bufferoverflowhigh -formatconst@*/
#if defined(TIGER_64_BIT)
      sprintf(outhash,                               /* known to fit  */
	      GPGFORMAT,
	      (sh_word32)(res[0]>>32), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[1]>>32), 
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[2]>>32), 
	      (sh_word32)(res[2]) );
#else
      sprintf(outhash,                               /* known to fit  */
	      GPGFORMAT,
	      (sh_word32)(res[1]), 
	      (sh_word32)(res[0]), 
	      (sh_word32)(res[3]), 
	      (sh_word32)(res[2]), 
	      (sh_word32)(res[5]), 
	      (sh_word32)(res[4]) );
#endif
      /*@+bufferoverflowhigh@*/
      outhash[48 + 6] = '\0';
    }
  else
    {
      /*@-bufferoverflowhigh@*/
      sprintf(outhash,                               /* known to fit  */
	      _("00000000 00000000 00000000  00000000 00000000 00000000"));
      /*@+bufferoverflowhigh@*/
    }

  if (what == TIGER_FILE)
    len = sl_strlen (filename) + 2 + 48 + 6;
  else
    len = 48 + 6;

  out = SH_ALLOC(len + 1);

  if (what == TIGER_FILE)
    {
      (void) sl_strlcpy (out, filename, len+1);
      (void) sl_strlcat (out,  _(": "), len+1);
      (void) sl_strlcat (out,  outhash, len+1);
    }
  else
    {
      (void) sl_strlcpy (out,  outhash, len+1);
    }
  SL_RETURN( out, _("sh_tiger_hash_gpg"));
}


UINT32 * sh_tiger_hash_uint32 (char * filename, 
			       TigerType what, 
			       UINT64 Length)
{
#if defined(TIGER_64_BIT)
  word64 * res;
#else
  sh_word32 * res;
#endif

  static UINT32 out[6];

  SL_ENTER(_("sh_tiger_hash_uint32"));

  out[0] = 0; out[1] = 0; out[2] = 0;
  out[3] = 0; out[4] = 0; out[5] = 0;

  res = sh_tiger_hash_val (filename,  what,  Length, 0);

  if (res != NULL)
    {
#if defined(TIGER_64_BIT)
	out[0] =  (UINT32)(res[0]>>32); 
	out[1] =  (UINT32)(res[0]);
	out[2] =  (UINT32)(res[1]>>32); 
	out[3] =  (UINT32)(res[1]); 
	out[4] =  (UINT32)(res[2]>>32); 
	out[5] =  (UINT32)(res[2]);
#else
	out[0] =  (UINT32)(res[1]); 
	out[1] =  (UINT32)(res[0]);
	out[2] =  (UINT32)(res[3]); 
	out[3] =  (UINT32)(res[2]); 
	out[4] =  (UINT32)(res[5]); 
	out[5] =  (UINT32)(res[4]);
#endif
    }

  SL_RETURN(out, _("sh_tiger_hash_uint32"));
}
  



